/** @file   dynamicevent.cpp
 * @brief   Implementation of DynamicEvent class
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/07/27 20:31:51 $
 * @author  Tomi Lamminsaari
 */

#include "dynamicevent.h"
#include "www_map.h"
#include "soundsamples.h"
#include "warglobals.h"
#include "spawnevent.h"
#include "modifyevent.h"
#include "scatterevent.h"
#include "controlevent.h"
#include "player.h"
#include <string>
#include "settings.h"
#include "www_assert.h"
using std::vector;
using std::ifstream;
using std::string;
using namespace eng2d;

namespace WeWantWar {


//********************************************************************
//                                                                   *
//      Static members and constants                                 *
//                                                                   *
//********************************************************************

const int DynamicEvent::KBitmaskEasy;
const int DynamicEvent::KBitmaskNormal;
const int DynamicEvent::KBitmaskHard;


//********************************************************************
//                                                                   *
//      Constructors, destructor and operators                       *
//                                                                   *
//********************************************************************

/** Constructs new instance DynamicEvent
 */
DynamicEvent::DynamicEvent() :
  m_status( WAITING ),
  m_counter( -1 ),
  m_x1( -1 ),
  m_y1( -1 ),
  m_x2( -2 ),
  m_y2( -2 ),
  m_difficulty( KBitmaskEasy | KBitmaskNormal | KBitmaskHard ),
  m_sound( -1 )
{
}



/** Destructor
 */
DynamicEvent::~DynamicEvent()
{
  // Delete all the inner events
  for (int i=0; i < m_innerEvents.size(); i++) {
    delete m_innerEvents.at(i);
  }
  m_innerEvents.clear();
}



//********************************************************************
//                                                                   *
//      Public interface                                             *
//                                                                   *
//********************************************************************



/** Default implementation does nothing.
 */
void DynamicEvent::activate()
{
  if ( m_status == COMPLETE ) {
    return;
  }
  
  m_status = ACTIVE;
  for (int i=0; i < m_innerEvents.size(); i++) {
    if ( m_innerEvents.at(i) != 0 ) {
      m_innerEvents.at(i)->activate();
    }
  }
}



/** Wakes up this event
 */
void DynamicEvent::wakeup()
{
  if ( m_status == SLEEP ) {
    m_status = ACTIVE;
  }
}



/** Sets this event asleep.
 */
void DynamicEvent::sleep()
{
  m_status = SLEEP;
}



/** Reads the data of this event.
 */
int DynamicEvent::readData( std::istream& fin )
{
  LOG_MESSAGE( "Reading Dynamic-event" );
  // The "<event>" - tag has already been read.
  string tmp1;
  string tmp2;
  string tmp3;
  
  // Read tl: - tag
  fin >> tmp1 >> tmp2 >> tmp3;
  if ( tmp1 != "tl:" ) {
    return -1;
  }
  m_x1 = atoi( tmp2.c_str() ) * 32;
  m_y1 = atoi( tmp3.c_str() ) * 32;
  
  // Read br: - tag
  fin >> tmp1 >> tmp2 >> tmp3;
  if ( tmp1 != "br:" ) {
    return -1;
  }
  m_x2 = atoi( tmp2.c_str() ) * 32;
  m_y2 = atoi( tmp3.c_str() ) * 32;
  
  // Read the rest of the event.
  while ( true ) {
    if ( fin.eof() ) {
      break;
    }
    
    fin >> tmp1;
    if ( tmp1 == "</event>" ) {
      return 0;
      
    } else if ( tmp1 == "<spawn>" ) {
      // Read spawn-event and place it to be our inner element.
      SpawnEvent* pSE = new SpawnEvent();
      if ( pSE->readData( fin ) != 0 ) {
        delete pSE;
        pSE = 0;
        return -1;
      }
      m_innerEvents.push_back( pSE );
      
    } else if ( tmp1 == "<modify>" ) {
      // Read the modify-event and place it to be out inner element.
      ModifyEvent* pME = new ModifyEvent();
      if ( pME->readData( fin ) != 0 ) {
        delete pME;
        pME = 0;
        return -1;
      }
      m_innerEvents.push_back( pME );
      

    } else if ( tmp1 == "<scatter>" ) {
      // Read the Scatter-event.
      ScatterEvent* pSE = new ScatterEvent();
      if ( pSE->readData( fin ) != 0 ) {
        delete pSE;
        pSE = 0;
        return -1;
      }
      m_innerEvents.push_back( pSE );
      
    } else if ( tmp1 == "<control>" ) {
      // Read the Control-event
      ControlEvent* pCE = new ControlEvent();
      if ( pCE->readData( fin ) != 0 ) {
        delete pCE;
        return -1;
      }
      m_innerEvents.push_back( pCE );
      
    } else if ( tmp1 == "#" ) {
      fin.ignore( 4096, '\n' );
      
    }
  }
  
  return -1;
}



/** Updates this event.
 */
void DynamicEvent::update()
{
  if ( m_status == WAITING ) {
    // We're waiting until the user goes inside the specified area.
    Vec2D pos = WarGlobals::pObjTable->pPlayer->position();
    if ( pos.x() > m_x1 && pos.x() < m_x2 && pos.y() > m_y1 && pos.y() < m_y2 ) {
      // Player is inside the activation area.
      this->activate();
    }
    
  } else if ( m_status == ACTIVE ) {
    bool allDone = true;
    
    for (int i=0; i < m_innerEvents.size(); i++) {
      if ( m_innerEvents.at(i) != 0 ) {
        m_innerEvents.at(i)->update();
        if ( m_innerEvents.at(i)->status() == ACTIVE ) {
          allDone = false;
        }
      }
    }
    if ( allDone ) {
      m_status = COMPLETE;
    }
  }
}



/** Sets the status
 */
void DynamicEvent::status( Status s )
{
  m_status = s;
}


bool DynamicEvent::isOkToActivate() const
{
  switch (Settings::difficultyLevel) {
    case (Settings::KEasy): {
      return ((m_difficulty & KBitmaskEasy) != 0) ? true : false;
    }
    case (Settings::KNormal): {
      return ((m_difficulty & KBitmaskNormal) != 0) ? true : false;
    }
    case (Settings::KHard): {
      return ((m_difficulty & KBitmaskHard) != 0) ? true : false;
    }
    default: {
      return false;
    }
  }
}
} // end of namespace

